"use client";
import React from "react";
import { Formik, Form, Field } from "formik";
import * as Yup from "yup";
import { useSubmitSurvayMutation } from "@/services/Api";
import toast from "react-hot-toast";

// Dynamic questions array
const questions = [
  {
    id: "name",
    label: "What is your name?",
    type: "text",
    validation: Yup.string().required("Please enter your name"),
  },
  {
    id: "email",
    label: "What is your email address?",
    type: "email",
    validation: Yup.string()
      .email("Please enter a valid email address")
      .required("Please enter your email address"),
  },
  {
    id: "q1",
    label: "Which age range best describes you?",
    type: "radio",
    options: ["Under 18", "18–22", "23–28", "29–35", "36–45", "46+"],
    validation: Yup.string().required("Please select an age range"),
    singleSelect: true,
  },
  {
    id: "q2",
    label: "What are you currently doing?",
    type: "radio",
    options: [
      "High school student",
      "College student",
      "Working part-time",
      "Working full-time",
      "Stay-at-home parent",
      "Exploring new career options",
      "Other",
    ],
    validation: Yup.string().required("Please select an option"),
    singleSelect: true,
    hasOtherText: true,
  },
  {
    id: "q3",
    label: "What excites you most about getting your real estate license?",
    type: "checkbox",
    options: [
      "The freedom to create my own schedule",
      "The potential to earn unlimited income",
      "The chance to make an impact and help others",
      "The ability to build long-term wealth",
      "The purpose and pride of building something of my own",
      "The thrill of learning a powerful new skill",
      "Other",
    ],
    validation: Yup.array()
      .min(1, "Please select at least one option")
      .max(3, "You can select up to three options"),
    singleSelect: false,
    hasOtherText: true,
  },
  {
    id: "q4",
    label:
      "One year from now, what would make you feel proud of your decision to get licensed?",
    type: "textarea",
    validation: Yup.string().required("Please provide an answer"),
  },
  {
    id: "q5",
    label:
      "What are you dreaming of creating through your real estate journey?",
    type: "textarea",
    validation: Yup.string().required("Please provide an answer"),
  },
  {
    id: "q6",
    label:
      "What qualities are most important to you in the real estate school you choose?",
    type: "checkbox",
    options: [
      "Clear, engaging instruction",
      "A learning experience that’s fun and exciting",
      "Supportive mentorship and guidance",
      "Practical tools to help me succeed after the exam",
      "A community of like-minded go-getters",
      "A course that connects to real-life goals, not just passing a test",
      "Other",
    ],
    validation: Yup.array()
      .min(1, "Please select at least one option")
      .max(3, "You can select up to three options"),
    singleSelect: false,
    hasOtherText: true,
  },
  {
    id: "q7",
    label:
      "How appealing is the idea of a real estate course that feels like watching a compelling TV series – with all content delivered through video instead of reading from a textbook?",
    type: "radio",
    options: ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10"],
    validation: Yup.string().required("Please select a rating"),
    singleSelect: true,
  },
  {
    id: "q8",
    label:
      "If you had a clear blueprint with ongoing support throughout the course – including answers to your questions, help with brokerage placement, and a step-by-step guide for your first 5 years – how confident would you feel about building a successful real estate career?",
    type: "radio",
    options: ["1", "2", "3", "4", "5", "6", "7", "8", "9", "10"],
    validation: Yup.string().required("Please select a rating"),
    singleSelect: true,
  },
  {
    id: "q9",
    label:
      "What would make your real estate learning journey feel exciting and energizing every step of the way?",
    type: "checkbox",
    options: [
      "Knowing there’s a clear step-by-step path to follow",
      "Learning through fun, interactive video content",
      "Being part of a supportive and inspiring community",
      "Having a mentor who believes in me",
      "Seeing real-world success stories that I can model",
      "Feeling like I’m building something that matters",
      "Other",
    ],
    validation: Yup.array()
      .min(1, "Please select at least one option")
      .max(3, "You can select up to three options"),
    singleSelect: false,
    hasOtherText: true,
  },
  {
    id: "q10",
    label:
      "What inspires you most about beginning this new chapter in real estate?",
    type: "radio",
    options: [
      "The chance to take control of my time and freedom",
      "The opportunity to grow my income and wealth",
      "The ability to make a difference and help people",
      "The pride of starting something meaningful on my own",
      "The excitement of mastering a new skill and industry",
      "The vision I have for a bigger, better future",
    ],
    validation: Yup.string().required("Please select an option"),
    singleSelect: true,
  },
];

// Create Yup validation schema dynamically
const createValidationSchema = () => {
  const schema = {};
  questions.forEach((question) => {
    schema[question.id] = question.validation;
    if (question.hasOtherText && question.options?.includes("Other")) {
      schema[`${question.id}_other`] = Yup.string().when(question.id, {
        is: (value) =>
          question.type === "radio"
            ? value === "Other"
            : Array.isArray(value) && value.includes("Other"),
        then: (schema) =>
          schema.required("Please specify your answer for Other"),
        otherwise: (schema) => schema.optional(),
      });
    }
  });
  return Yup.object().shape(schema);
};

const SurvayForm = () => {
  const [submitSurvey, { isLoading }] = useSubmitSurvayMutation();

  // Initial values for Formik
  const initialValues = questions.reduce((acc, question) => {
    acc[question.id] = question.type === "checkbox" ? [] : "";
    if (question.hasOtherText && question.options?.includes("Other")) {
      acc[`${question.id}_other`] = "";
    }
    return acc;
  }, {});

  const handleSubmit = async (values, { setSubmitting, resetForm }) => {
    try {
      const response = await submitSurvey({ data: values }).unwrap();
      toast.success("Survey submitted successfully!", {
        position: "top-center",
        autoClose: 3000,
      });
      resetForm();
      console.log("API Response:", response);
    } catch (error) {
      toast.error(error?.data?.message || "Failed to submit survey. Please try again.", {
        position: "top-center",
        autoClose: 3000,
      });
      console.error("Submission Error:", error);
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <div className="bg-[#000] w-full min-h-[100vh] coming-soon-page px-3 flex flex-col justify-center">
      <div className="max-w-7xl py-10 px-6 mx-auto z-10">
        <h1 className="text-white text-3xl font-semibold text-center mb-16">
          “Find Your Path” Survey
        </h1>
        <div className="flex gap-8">
          <div className="w-full">
            <Formik
              initialValues={initialValues}
              validationSchema={createValidationSchema()}
              onSubmit={handleSubmit}
            >
              {({ errors, touched, values, isSubmitting }) => (
                <Form>
                  <h2 className="text-white text-xl font-semibold mb-3">
                    Discover the real estate journey that aligns with your
                    ambition.
                  </h2>
                  {questions.map((question, index) => (
                    <div key={question.id} className="mb-6">
                      <label className="text-white text-xl font-semibold w-full inline-block">
                        {`${index + 1}: ${question.label}`}
                      </label>
                      <span className="text-base text-white">
                        {question.type === "checkbox"
                          ? "(Choose up to 3)"
                          : question.type === "radio" && question.singleSelect
                          ? "(Select One)"
                          : ""}
                      </span>
                      {question.type === "textarea" ? (
                        <Field
                          as="textarea"
                          name={question.id}
                          className="w-full py-1.5 pr-3 px-2 mt-3 rounded-md text-base text-white bg-transparent border placeholder:text-gray-400 border-[#E3E3E3] focus:border-golden outline-none"
                          rows={5}
                          placeholder="Enter Your Answer...."
                        />
                      ) : question.type === "text" || question.type === "email" ? (
                        <Field
                          type={question.type}
                          name={question.id}
                          className="w-full py-1.5 pr-3 px-2 mt-3 rounded-md text-base text-white bg-transparent border placeholder:text-gray-400 border-[#E3E3E3] focus:border-golden outline-none"
                          placeholder={`Enter your ${question.type === "email" ? "email address" : "name"}...`}
                        />
                      ) : (
                        <ul className="list-none">
                          <li>
                            <div className="flex flex-col pt-3 ps-2">
                              {question.options?.map((option) => (
                                <label
                                  key={option}
                                  className="custom-radio flex gap-2 items-center text-white"
                                >
                                  <Field
                                    type={question.type}
                                    name={question.id}
                                    value={option}
                                    disabled={
                                      question.type === "checkbox" &&
                                      values[question.id].length >= 3 &&
                                      !values[question.id].includes(option)
                                    }
                                  />
                                  {option}
                                </label>
                              ))}
                              {question.hasOtherText &&
                                question.options?.includes("Other") &&
                                ((question.type === "radio" &&
                                  values[question.id] === "Other") ||
                                  (question.type === "checkbox" &&
                                    values[question.id].includes("Other"))) && (
                                  <Field
                                    as="textarea"
                                    name={`${question.id}_other`}
                                    className="w-full py-1.5 pr-3 px-2 mt-3 rounded-md text-base text-white bg-transparent border placeholder:text-gray-400 border-[#E3E3E3] focus:border-golden outline-none"
                                    rows={5}
                                    placeholder="Please specify your answer..."
                                  />
                                )}
                            </div>
                          </li>
                        </ul>
                      )}
                      {errors[question.id] && touched[question.id] && (
                        <div className="text-red-500 text-sm mt-1">
                          {errors[question.id]}
                        </div>
                      )}
                      {question.hasOtherText &&
                        question.options?.includes("Other") &&
                        errors[`${question.id}_other`] &&
                        touched[`${question.id}_other`] && (
                          <div className="text-red-500 text-sm mt-1">
                            {errors[`${question.id}_other`]}
                          </div>
                        )}
                    </div>
                  ))}
                  <button
                    type="submit"
                    className="bg-golden text-white py-2 px-3 rounded-md text-lg font-medium disabled:opacity-50"
                    disabled={isSubmitting || isLoading}
                  >
                    {isSubmitting || isLoading ? "Submitting..." : "Submit"}
                  </button>
                </Form>
              )}
            </Formik>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SurvayForm;